/****************************************************************************
 *   $Id:: uart.c 5523 2010-11-04 15:44:08Z nxp21346                        $
 *   Project: NXP LPC17xx I2C example
 *
 *   Description:
 *     UART API file for NXP LPC177x_8x Family Microprocessors
 *
 ****************************************************************************
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * products. This software is supplied "AS IS" without any warranties.
 * NXP Semiconductors assumes no responsibility or liability for the
 * use of the software, conveys no license or title under any patent,
 * copyright, or mask work right to the product. NXP Semiconductors
 * reserves the right to make changes in the software without
 * notification. NXP Semiconductors also make no representation or
 * warranty that such application will be suitable for the specified
 * use without further testing or modification.
****************************************************************************/

#include "LPC177x_8x.h"
#include "type.h"
#include "uart.h"

volatile uint32_t UART0Status, UART1Status, UART2Status, UART3Status, UART4Status;
volatile uint8_t UART0TxEmpty = 1, UART1TxEmpty = 1, UART2TxEmpty = 1, UART3TxEmpty = 1, UART4TxEmpty = 1;
volatile uint8_t UART0Buffer[BUFSIZE], UART1Buffer[BUFSIZE], UART2Buffer[BUFSIZE], UART3Buffer[BUFSIZE], UART4Buffer[BUFSIZE];
volatile uint32_t UART0Count = 0, UART1Count = 0, UART2Count = 0, UART3Count = 0, UART4Count = 0;

/*****************************************************************************
** Function name:		UART0_IRQHandler
**
** Descriptions:		UART0 interrupt handler
**
** parameters:			None
** Returned value:		None
** 
*****************************************************************************/
void UART0_IRQHandler (void) 
{
  uint8_t IIRValue, LSRValue;
  uint8_t Dummy = Dummy;
	
  IIRValue = LPC_UART0->IIR;
    
  IIRValue >>= 1;			/* skip pending bit in IIR */
  IIRValue &= 0x07;			/* check bit 1~3, interrupt identification */
  if ( IIRValue == IIR_RLS )		/* Receive Line Status */
  {
	LSRValue = LPC_UART0->LSR;
	/* Receive Line Status */
	if ( LSRValue & (LSR_OE|LSR_PE|LSR_FE|LSR_RXFE|LSR_BI) )
	{
	  /* There are errors or break interrupt */
	  /* Read LSR will clear the interrupt */
	  UART0Status = LSRValue;
	  Dummy = LPC_UART0->RBR;		/* Dummy read on RX to clear 
							interrupt, then bail out */
	  return;
	}
	if ( LSRValue & LSR_RDR )	/* Receive Data Ready */			
	{
	  /* If no error on RLS, normal ready, save into the data buffer. */
	  /* Note: read RBR will clear the interrupt */
	  UART0Buffer[UART0Count] = LPC_UART0->RBR;
	  UART0Count++;
	  if ( UART0Count == BUFSIZE )
	  {
		UART0Count = 0;		/* buffer overflow */
	  }	
	}
  }
  else if ( IIRValue == IIR_RDA )	/* Receive Data Available */
  {
	/* Receive Data Available */
	UART0Buffer[UART0Count] = LPC_UART0->RBR;
	UART0Count++;
	if ( UART0Count == BUFSIZE )
	{
	  UART0Count = 0;		/* buffer overflow */
	}
  }
  else if ( IIRValue == IIR_CTI )	/* Character timeout indicator */
  {
	/* Character Time-out indicator */
	UART0Status |= 0x100;		/* Bit 9 as the CTI error */
  }
  else if ( IIRValue == IIR_THRE )	/* THRE, transmit holding register empty */
  {
	/* THRE interrupt */
	LSRValue = LPC_UART0->LSR;		/* Check status in the LSR to see if
									valid data in U0THR or not */
	if ( LSRValue & LSR_THRE )
	{
	  UART0TxEmpty = 1;
	}
	else
	{
	  UART0TxEmpty = 0;
	}
  }
}

/*****************************************************************************
** Function name:		UART1_IRQHandler
**
** Descriptions:		UART1 interrupt handler
**
** parameters:			None
** Returned value:		None
** 
*****************************************************************************/
void UART1_IRQHandler (void) 
{
  uint8_t IIRValue, LSRValue;
  uint8_t Dummy = Dummy;
	
  IIRValue = LPC_UART1->IIR;
    
  IIRValue >>= 1;			/* skip pending bit in IIR */
  IIRValue &= 0x07;			/* check bit 1~3, interrupt identification */
  if ( IIRValue == IIR_RLS )		/* Receive Line Status */
  {
	LSRValue = LPC_UART1->LSR;
	/* Receive Line Status */
	if ( LSRValue & (LSR_OE|LSR_PE|LSR_FE|LSR_RXFE|LSR_BI) )
	{
	  /* There are errors or break interrupt */
	  /* Read LSR will clear the interrupt */
	  UART1Status = LSRValue;
	  Dummy = LPC_UART1->RBR;		/* Dummy read on RX to clear 
								interrupt, then bail out */
	  return;
	}
	if ( LSRValue & LSR_RDR )	/* Receive Data Ready */			
	{
	  /* If no error on RLS, normal ready, save into the data buffer. */
	  /* Note: read RBR will clear the interrupt */
	  UART1Buffer[UART1Count] = LPC_UART1->RBR;
	  UART1Count++;
	  if ( UART1Count == BUFSIZE )
	  {
		UART1Count = 0;		/* buffer overflow */
	  }	
	}
  }
  else if ( IIRValue == IIR_RDA )	/* Receive Data Available */
  {
	/* Receive Data Available */
	UART1Buffer[UART1Count] = LPC_UART1->RBR;
	UART1Count++;
	if ( UART1Count == BUFSIZE )
	{
	  UART1Count = 0;		/* buffer overflow */
	}
  }
  else if ( IIRValue == IIR_CTI )	/* Character timeout indicator */
  {
	/* Character Time-out indicator */
	UART1Status |= 0x100;		/* Bit 9 as the CTI error */
  }
  else if ( IIRValue == IIR_THRE )	/* THRE, transmit holding register empty */
  {
	/* THRE interrupt */
	LSRValue = LPC_UART1->LSR;		/* Check status in the LSR to see if
								valid data in U0THR or not */
	if ( LSRValue & LSR_THRE )
	{
	  UART1TxEmpty = 1;
	}
	else
	{
	  UART1TxEmpty = 0;
	}
  }
}

/*****************************************************************************
** Function name:		UART2_IRQHandler
**
** Descriptions:		UART2 interrupt handler
**
** parameters:			None
** Returned value:		None
** 
*****************************************************************************/
void UART2_IRQHandler (void) 
{
  uint8_t IIRValue, LSRValue;
  uint8_t Dummy = Dummy;
	
  IIRValue = LPC_UART2->IIR;
    
  IIRValue >>= 1;			/* skip pending bit in IIR */
  IIRValue &= 0x07;			/* check bit 1~3, interrupt identification */
  if ( IIRValue == IIR_RLS )		/* Receive Line Status */
  {
	LSRValue = LPC_UART2->LSR;
	/* Receive Line Status */
	if ( LSRValue & (LSR_OE|LSR_PE|LSR_FE|LSR_RXFE|LSR_BI) )
	{
	  /* There are errors or break interrupt */
	  /* Read LSR will clear the interrupt */
	  UART2Status = LSRValue;
	  Dummy = LPC_UART2->RBR;		/* Dummy read on RX to clear 
								interrupt, then bail out */
	  return;
	}
	if ( LSRValue & LSR_RDR )	/* Receive Data Ready */			
	{
	  /* If no error on RLS, normal ready, save into the data buffer. */
	  /* Note: read RBR will clear the interrupt */
	  UART2Buffer[UART2Count] = LPC_UART2->RBR;
	  UART2Count++;
	  if ( UART2Count == BUFSIZE )
	  {
		UART2Count = 0;		/* buffer overflow */
	  }	
	}
  }
  else if ( IIRValue == IIR_RDA )	/* Receive Data Available */
  {
	/* Receive Data Available */
	UART2Buffer[UART2Count] = LPC_UART2->RBR;
	UART2Count++;
	if ( UART2Count == BUFSIZE )
	{
	  UART2Count = 0;		/* buffer overflow */
	}
  }
  else if ( IIRValue == IIR_CTI )	/* Character timeout indicator */
  {
	/* Character Time-out indicator */
	UART2Status |= 0x100;		/* Bit 9 as the CTI error */
  }
  else if ( IIRValue == IIR_THRE )	/* THRE, transmit holding register empty */
  {
	/* THRE interrupt */
	LSRValue = LPC_UART2->LSR;		/* Check status in the LSR to see if
								valid data in U0THR or not */
	if ( LSRValue & LSR_THRE )
	{
	  UART2TxEmpty = 1;
	}
	else
	{
	  UART2TxEmpty = 0;
	}
  }
}

/*****************************************************************************
** Function name:		UART3_IRQHandler
**
** Descriptions:		UART3 interrupt handler
**
** parameters:			None
** Returned value:		None
** 
*****************************************************************************/
void UART3_IRQHandler (void) 
{
  uint8_t IIRValue, LSRValue;
  uint8_t Dummy = Dummy;
	
  IIRValue = LPC_UART3->IIR;
    
  IIRValue >>= 1;			/* skip pending bit in IIR */
  IIRValue &= 0x07;			/* check bit 1~3, interrupt identification */
  if ( IIRValue == IIR_RLS )		/* Receive Line Status */
  {
	LSRValue = LPC_UART3->LSR;
	/* Receive Line Status */
	if ( LSRValue & (LSR_OE|LSR_PE|LSR_FE|LSR_RXFE|LSR_BI) )
	{
	  /* There are errors or break interrupt */
	  /* Read LSR will clear the interrupt */
	  UART3Status = LSRValue;
	  Dummy = LPC_UART3->RBR;		/* Dummy read on RX to clear 
								interrupt, then bail out */
	  return;
	}
	if ( LSRValue & LSR_RDR )	/* Receive Data Ready */			
	{
	  /* If no error on RLS, normal ready, save into the data buffer. */
	  /* Note: read RBR will clear the interrupt */
	  UART3Buffer[UART3Count] = LPC_UART3->RBR;
	  UART3Count++;
	  if ( UART3Count == BUFSIZE )
	  {
		UART3Count = 0;		/* buffer overflow */
	  }	
	}
  }
  else if ( IIRValue == IIR_RDA )	/* Receive Data Available */
  {
	/* Receive Data Available */
	UART3Buffer[UART3Count] = LPC_UART3->RBR;
	UART3Count++;
	if ( UART3Count == BUFSIZE )
	{
	  UART3Count = 0;		/* buffer overflow */
	}
  }
  else if ( IIRValue == IIR_CTI )	/* Character timeout indicator */
  {
	/* Character Time-out indicator */
	UART3Status |= 0x100;		/* Bit 9 as the CTI error */
  }
  else if ( IIRValue == IIR_THRE )	/* THRE, transmit holding register empty */
  {
	/* THRE interrupt */
	LSRValue = LPC_UART3->LSR;		/* Check status in the LSR to see if
								valid data in U0THR or not */
	if ( LSRValue & LSR_THRE )
	{
	  UART3TxEmpty = 1;
	}
	else
	{
	  UART3TxEmpty = 0;
	}
  }
}

/*****************************************************************************
** Function name:		UART4_IRQHandler
**
** Descriptions:		UART4 interrupt handler
**
** parameters:			None
** Returned value:		None
** 
*****************************************************************************/
void UART4_IRQHandler (void) 
{
  uint8_t IIRValue, LSRValue;
  uint8_t Dummy = Dummy;
	
  IIRValue = LPC_UART4->IIR;
    
  IIRValue >>= 1;			/* skip pending bit in IIR */
  IIRValue &= 0x07;			/* check bit 1~3, interrupt identification */
  if ( IIRValue == IIR_RLS )		/* Receive Line Status */
  {
	LSRValue = LPC_UART4->LSR;
	/* Receive Line Status */
	if ( LSRValue & (LSR_OE|LSR_PE|LSR_FE|LSR_RXFE|LSR_BI) )
	{
	  /* There are errors or break interrupt */
	  /* Read LSR will clear the interrupt */
	  UART4Status = LSRValue;
	  Dummy = LPC_UART4->RBR;		/* Dummy read on RX to clear 
								interrupt, then bail out */
	  return;
	}
	if ( LSRValue & LSR_RDR )	/* Receive Data Ready */			
	{
	  /* If no error on RLS, normal ready, save into the data buffer. */
	  /* Note: read RBR will clear the interrupt */
	  UART4Buffer[UART4Count] = LPC_UART4->RBR;
	  UART4Count++;
	  if ( UART4Count == BUFSIZE )
	  {
		UART4Count = 0;		/* buffer overflow */
	  }	
	}
  }
  else if ( IIRValue == IIR_RDA )	/* Receive Data Available */
  {
	/* Receive Data Available */
	UART4Buffer[UART4Count] = LPC_UART4->RBR;
	UART4Count++;
	if ( UART4Count == BUFSIZE )
	{
	  UART4Count = 0;		/* buffer overflow */
	}
  }
  else if ( IIRValue == IIR_CTI )	/* Character timeout indicator */
  {
	/* Character Time-out indicator */
	UART4Status |= 0x100;		/* Bit 9 as the CTI error */
  }
  else if ( IIRValue == IIR_THRE )	/* THRE, transmit holding register empty */
  {
	/* THRE interrupt */
	LSRValue = LPC_UART4->LSR;		/* Check status in the LSR to see if
								valid data in U0THR or not */
	if ( LSRValue & LSR_THRE )
	{
	  UART4TxEmpty = 1;
	}
	else
	{
	  UART4TxEmpty = 0;
	}
  }
}

/*****************************************************************************
** Function name:		ModemInit
**
** Descriptions:		Only UART1 has modem feature. Initialize UART1 port as 
**                      modem, setup pin select.
**
** parameters:			None
** Returned value:		None
** 
*****************************************************************************/
void ModemInit( void )
{
#if 1
  LPC_IOCON->P0_20 &= ~0x07;    /* UART I/O config */
  LPC_IOCON->P0_20 |= 0x01;     /* UART DTR */
  LPC_IOCON->P0_17 &= ~0x07;    /* UART I/O config */
  LPC_IOCON->P0_17 |= 0x01;     /* UART CTS */
  LPC_IOCON->P0_22 &= ~0x07;    /* UART I/O config */
  LPC_IOCON->P0_22 |= 0x01;     /* UART RTS */

  LPC_IOCON->P0_19 &= ~0x07;    /* UART I/O config */
  LPC_IOCON->P0_19 |= 0x01;     /* UART DSR */
  LPC_IOCON->P0_18 &= ~0x07;    /* UART I/O config */
  LPC_IOCON->P0_18 |= 0x01;     /* UART DCD */
  LPC_IOCON->P0_21 &= ~0x07;    /* UART I/O config */
  LPC_IOCON->P0_21 |= 0x01;     /* UART RI */
#endif

#if 0
  LPC_IOCON->P2_5  &= ~0x07;    /* UART I/O config */
  LPC_IOCON->P2_5  |= 0x02;     /* UART DTR */
  LPC_IOCON->P2_2  &= ~0x07;    /* UART I/O config */
  LPC_IOCON->P2_2  |= 0x02;     /* UART CTS */
  LPC_IOCON->P2_7  &= ~0x07;    /* UART I/O config */
  LPC_IOCON->P2_7  |= 0x02;     /* UART RTS */

  LPC_IOCON->P2_4  &= ~0x07;    /* UART I/O config */
  LPC_IOCON->P2_4  |= 0x02;     /* UART DSR */
  LPC_IOCON->P2_3  &= ~0x07;    /* UART I/O config */
  LPC_IOCON->P2_3  |= 0x02;     /* UART DCD */
  LPC_IOCON->P2_6  &= ~0x07;    /* UART I/O config */
  LPC_IOCON->P2_6  |= 0x02;     /* UART RI */
#endif

#if 0
  LPC_IOCON->P2_5  &= ~0x07;    /* UART I/O config */
  LPC_IOCON->P2_5  |= 0x02;     /* UART DTR */
  LPC_IOCON->P2_8  &= ~0x07;    /* UART I/O config */
  LPC_IOCON->P2_8  |= 0x03;     /* UART CTS */
  LPC_IOCON->P0_6  &= ~0x07;    /* UART I/O config */
  LPC_IOCON->P0_6  |= 0x04;     /* UART RTS */

  LPC_IOCON->P2_4  &= ~0x07;    /* UART I/O config */
  LPC_IOCON->P2_4  |= 0x02;     /* UART DSR */
  LPC_IOCON->P2_3  &= ~0x07;    /* UART I/O config */
  LPC_IOCON->P2_3  |= 0x02;     /* UART DCD */
  LPC_IOCON->P2_6  &= ~0x07;    /* UART I/O config */
  LPC_IOCON->P2_6  |= 0x02;     /* UART RI */
#endif

#if 0
  LPC_IOCON->P3_21 &= ~0x07;    /* UART I/O config */
  LPC_IOCON->P3_21 |= 0x03;     /* UART DTR */
  LPC_IOCON->P3_18 &= ~0x07;    /* UART I/O config */
  LPC_IOCON->P3_18 |= 0x03;     /* UART CTS */
  LPC_IOCON->P3_30 &= ~0x07;    /* UART I/O config */
  LPC_IOCON->P3_30 |= 0x02;     /* UART RTS */

  LPC_IOCON->P3_20 &= ~0x07;    /* UART I/O config */
  LPC_IOCON->P3_20 |= 0x03;     /* UART DSR */
  LPC_IOCON->P3_19 &= ~0x07;    /* UART I/O config */
  LPC_IOCON->P3_19 |= 0x03;     /* UART DCD */
  LPC_IOCON->P3_22 &= ~0x07;    /* UART I/O config */
  LPC_IOCON->P3_22 |= 0x03;     /* UART RI */
#endif

  LPC_UART1->MCR = 0xC0;          /* Enable Auto RTS and Auto CTS. */			
  return;
}

/*****************************************************************************
** Function name:		UARTInit
**
** Descriptions:		Initialize UART port, setup pin select,
**						clock, parity, stop bits, FIFO, etc.
**
** parameters:			portNum(0 or 1) and UART baudrate
** Returned value:		true or false, return false only if the 
**						interrupt handler can't be installed to the 
**						VIC table
** 
*****************************************************************************/
uint32_t UARTInit( uint32_t PortNum, uint32_t baudrate )
{
  uint32_t Fdiv;

  if ( PortNum == 0 )
  {
	LPC_SC->PCONP |= 0x00000008;
#if 0
    LPC_IOCON->P0_0  &= ~0x07;
    LPC_IOCON->P0_0  |= 4; /* U0_TXD @ P0.0 */
    LPC_IOCON->P0_1  &= ~0x07;
    LPC_IOCON->P0_1  |= 4; /* U0_RXD @ P0.1 */
#endif
#if 1
    LPC_IOCON->P0_2  &= ~0x07;
    LPC_IOCON->P0_2  |= 1; /* U0_TXD @ P0.2 */
    LPC_IOCON->P0_3  &= ~0x07;
    LPC_IOCON->P0_3  |= 1; /* U0_RXD @ P0.3 */
#endif

    LPC_UART0->LCR = 0x83;		/* 8 bits, no Parity, 1 Stop bit */
	Fdiv = ( SystemCoreClock / (LPC_SC->PCLKSEL&0x1F) / 16 ) / baudrate ;	/*baud rate */
    LPC_UART0->DLM = Fdiv / 256;							
    LPC_UART0->DLL = Fdiv % 256;
	LPC_UART0->LCR = 0x03;		/* DLAB = 0 */
    LPC_UART0->FCR = 0x07;		/* Enable and reset TX and RX FIFO. */

   	NVIC_EnableIRQ(UART0_IRQn);

    LPC_UART0->IER = IER_RBR | IER_THRE | IER_RLS;	/* Enable UART0 interrupt */
    return (TRUE);
  }
  else if ( PortNum == 1 )
  {
	LPC_SC->PCONP |= 0x00000010;
#if 0
    LPC_IOCON->P0_15 &= ~0x07;
    LPC_IOCON->P0_15 |= 1; /* U1_TXD @ P0.15 */
    LPC_IOCON->P0_16 &= ~0x07;
    LPC_IOCON->P0_16 |= 1; /* U1_RXD @ P0.16 */
#endif
#if 1
    LPC_IOCON->P3_16 &= ~0x07;
    LPC_IOCON->P3_16 |= 3; /* U1_TXD @ P3.16 */
    LPC_IOCON->P3_17 &= ~0x07;
    LPC_IOCON->P3_17 |= 3; /* U1_RXD @ P3.17 */
#endif
#if 0
    LPC_IOCON->P2_0  &= ~0x07;
    LPC_IOCON->P2_0  |= 2; /* U1_TXD @ P2.0 */
    LPC_IOCON->P2_1  &= ~0x07;
    LPC_IOCON->P2_1  |= 2; /* U1_RXD @ P2.1 */
#endif

    LPC_UART1->LCR = 0x83;		/* 8 bits, no Parity, 1 Stop bit */
	Fdiv = ( SystemCoreClock / (LPC_SC->PCLKSEL&0x1F) / 16 ) / baudrate ;	/*baud rate */
    LPC_UART1->DLM = Fdiv / 256;							
    LPC_UART1->DLL = Fdiv % 256;
	LPC_UART1->LCR = 0x03;		/* DLAB = 0 */
    LPC_UART1->FCR = 0x07;		/* Enable and reset TX and RX FIFO. */

   	NVIC_EnableIRQ(UART1_IRQn);

    LPC_UART1->IER = IER_RBR | IER_THRE | IER_RLS;	/* Enable UART1 interrupt */
    return (TRUE);
  }
  else if ( PortNum == 2 )
  {
	LPC_SC->PCONP |= 0x01000000;
#if 1
    LPC_IOCON->P0_10  &= ~0x07;
    LPC_IOCON->P0_10 |= 1; /* U2_TXD @ P0.10 */
    LPC_IOCON->P0_11  &= ~0x07;
    LPC_IOCON->P0_11 |= 1; /* U2_RXD @ P0.11 */
#endif
#if 0
    LPC_IOCON->P4_22  &= ~0x07;
    LPC_IOCON->P4_22 |= 2; /* U2_TXD @ P4.22 */
    LPC_IOCON->P4_23  &= ~0x07;
    LPC_IOCON->P4_23 |= 2; /* U2_RXD @ P4.23 */
#endif
#if 0
    LPC_IOCON->P2_8  &= ~0x07;
    LPC_IOCON->P2_8  |= 2; /* U2_TXD @ P2.8 */
    LPC_IOCON->P2_9  &= ~0x07;
    LPC_IOCON->P2_9  |= 2; /* U2_RXD @ P2.9 */
#endif

    LPC_UART2->LCR = 0x83;		/* 8 bits, no Parity, 1 Stop bit */
	Fdiv = ( SystemCoreClock / (LPC_SC->PCLKSEL&0x1F) / 16 ) / baudrate ;	/*baud rate */
    LPC_UART2->DLM = Fdiv / 256;							
    LPC_UART2->DLL = Fdiv % 256;
	LPC_UART2->LCR = 0x03;		/* DLAB = 0 */
    LPC_UART2->FCR = 0x07;		/* Enable and reset TX and RX FIFO. */

   	NVIC_EnableIRQ(UART2_IRQn);

    LPC_UART2->IER = IER_RBR | IER_THRE | IER_RLS;	/* Enable UART1 interrupt */
    return (TRUE);
  }
  else if ( PortNum == 3 )
  {
	LPC_SC->PCONP |= 0x02000000;
#if 1
    LPC_IOCON->P0_25  &= ~0x07;
    LPC_IOCON->P0_25 |= 3; /* U3_TXD @ P0.25 */
    LPC_IOCON->P0_26  &= ~0x07;
    LPC_IOCON->P0_26 |= 3; /* U3_RXD @ P0.26 */
#endif
#if 0
    LPC_IOCON->P0_0  &= ~0x07;
    LPC_IOCON->P0_0  |= 2; /* U3_TXD @ P0.0 */
    LPC_IOCON->P0_1  &= ~0x07;
    LPC_IOCON->P0_1  |= 2; /* U3_RXD @ P0.1 */
#endif
#if 0
    LPC_IOCON->P4_28  &= ~0x07;
    LPC_IOCON->P4_28 |= 2; /* U3_TXD @ P4.28 */
    LPC_IOCON->P4_29  &= ~0x07;
    LPC_IOCON->P4_29 |= 2; /* U3_RXD @ P4.29 */
#endif
#if 0
    LPC_IOCON->P0_2  &= ~0x07;
    LPC_IOCON->P0_2  |= 2; /* U3_TXD @ P0.2 */
    LPC_IOCON->P0_3  &= ~0x07;
    LPC_IOCON->P0_3  |= 2; /* U3_RXD @ P0.3 */
#endif

    LPC_UART3->LCR = 0x83;		/* 8 bits, no Parity, 1 Stop bit */
	Fdiv = ( SystemCoreClock / (LPC_SC->PCLKSEL&0x1F) / 16 ) / baudrate ;	/*baud rate */
    LPC_UART3->DLM = Fdiv / 256;							
    LPC_UART3->DLL = Fdiv % 256;
	LPC_UART3->LCR = 0x03;		/* DLAB = 0 */
    LPC_UART3->FCR = 0x07;		/* Enable and reset TX and RX FIFO. */

   	NVIC_EnableIRQ(UART3_IRQn);

    LPC_UART3->IER = IER_RBR | IER_THRE | IER_RLS;	/* Enable UART1 interrupt */
    return (TRUE);
  }
  else if ( PortNum == 4 )
  {
	LPC_SC->PCONP |= 0x00000100;
#if 1
    LPC_IOCON->P0_22 &= ~0x07;
    LPC_IOCON->P0_22 |= 3; /* U4_TXD @ P0.22 */
    LPC_IOCON->P2_9  &= ~0x07;
    LPC_IOCON->P2_9  |= 3; /* U4_RXD @ P2.9 */
#endif
#if 0
    LPC_IOCON->P1_29 &= ~0x07;
    LPC_IOCON->P1_29 |= 5; /* U4_TXD @ P1.29 */
    LPC_IOCON->P2_9  &= ~0x07;
    LPC_IOCON->P2_9  |= 3; /* U4_RXD @ P2.9 */
#endif
#if 0
    LPC_IOCON->P5_4  &= ~0x07;
    LPC_IOCON->P5_4  |= 4; /* U4_TXD @ P5.4 */
    LPC_IOCON->P5_3  &= ~0x07;
    LPC_IOCON->P5_3  |= 4; /* U4_RXD @ P5.3 */
#endif

    LPC_UART4->LCR = 0x83;		/* 8 bits, no Parity, 1 Stop bit */
	Fdiv = ( SystemCoreClock / (LPC_SC->PCLKSEL&0x1F) / 16 ) / baudrate ;	/*baud rate */
    LPC_UART4->DLM = Fdiv / 256;							
    LPC_UART4->DLL = Fdiv % 256;
	LPC_UART4->LCR = 0x03;		/* DLAB = 0 */
    LPC_UART4->FCR = 0x07;		/* Enable and reset TX and RX FIFO. */

   	NVIC_EnableIRQ(UART4_IRQn);

    LPC_UART4->IER = IER_RBR | IER_THRE | IER_RLS;	/* Enable UART1 interrupt */
    return (TRUE);
  }
  return( FALSE ); 
}

/*****************************************************************************
** Function name:		UARTSend
**
** Descriptions:		Send a block of data to the UART 0 port based
**						on the data length
**
** parameters:			portNum, buffer pointer, and data length
** Returned value:		None
** 
*****************************************************************************/
void UARTSend( uint32_t portNum, uint8_t *BufferPtr, uint32_t Length )
{
  if ( portNum == 0 )
  {
    while ( Length != 0 )
    {
	  /* THRE status, contain valid data */
	  while ( !(UART0TxEmpty & 0x01) );	
	  LPC_UART0->THR = *BufferPtr;
	  UART0TxEmpty = 0;	/* not empty in the THR until it shifts out */
	  BufferPtr++;
	  Length--;
	}
  }
  else if ( portNum == 1 )
  {
	while ( Length != 0 )
    {
	  /* THRE status, contain valid data */
	  while ( !(UART1TxEmpty & 0x01) );	
	  LPC_UART1->THR = *BufferPtr;
	  UART1TxEmpty = 0;	/* not empty in the THR until it shifts out */
	  BufferPtr++;
	  Length--;
    }
  }
  else if ( portNum == 2 )
  {
	while ( Length != 0 )
    {
	  /* THRE status, contain valid data */
	  while ( !(UART2TxEmpty & 0x01) );	
	  LPC_UART2->THR = *BufferPtr;
	  UART2TxEmpty = 0;	/* not empty in the THR until it shifts out */
	  BufferPtr++;
	  Length--;
    }
  }
  else if ( portNum == 3 )
  {
	while ( Length != 0 )
    {
	  /* THRE status, contain valid data */
	  while ( !(UART3TxEmpty & 0x01) );	
	  LPC_UART3->THR = *BufferPtr;
	  UART3TxEmpty = 0;	/* not empty in the THR until it shifts out */
	  BufferPtr++;
	  Length--;
    }
  }
  else if ( portNum == 4 )
  {
	while ( Length != 0 )
    {
	  /* THRE status, contain valid data */
	  while ( !(UART4TxEmpty & 0x01) );	
	  LPC_UART4->THR = *BufferPtr;
	  UART4TxEmpty = 0;	/* not empty in the THR until it shifts out */
	  BufferPtr++;
	  Length--;
    }
  }
  return;
}

								 /* Implementation of putchar (also used by printf function to output data)    */
int sendchar (int ch)  {                 /* Write character to Serial Port    */

  while (!(LPC_UART0->LSR & 0x20));

  return (LPC_UART0->THR = ch);
}


int getkey (void)  {                     /* Read character from Serial Port   */

  while (!(LPC_UART0->LSR & 0x01));

  return (LPC_UART0->RBR);
}


/******************************************************************************
**                            End Of File
******************************************************************************/
